/* $Id: sourceaz.c,v 1.10 1998/09/16 02:05:17 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: sourceaz.c,v 1.10 1998/09/16 02:05:17 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].source_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    if (nc == 0)
    {
	(void) fprintf(stderr, "No source found!\n");
	return -1;
    }

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_SOURCE_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group)
{
    /* 5 KHz sine, 6.4 KHz filter, with SUM input on */
    /* Amp scale set to zero so we get DC out */
    CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
    CHECK(e1432_set_range(hw, group, 10));
    CHECK(e1432_set_amp_scale(hw, group, 0));
    CHECK(e1432_set_filter_freq(hw, group, 6400));
    CHECK(e1432_set_sine_freq(hw, group, 5000));
    CHECK(e1432_set_source_cola(hw, group, E1432_SOURCE_COLA_OFF));
    CHECK(e1432_set_source_sum(hw, group, E1432_SOURCE_SUM_ON));

    /* Auto-zero the source */

    /* Normally, you could just call e1432_auto_zero(hw, group), and
       that would be sufficient.

       However, if the filter frequency programmed into the source
       channels changes, this is not sufficient.

       We set the filter frequency to 6400, rather than the default
       25600, so we must follow the more complicated procedure
       auto-zero the source.  First, set the source output mode to
       grounded.  This prevents any glitching at the source BNC, but
       it causes the source to drive the module's CALOUT line.  Then,
       set the source arm mode to manual, so the source never really
       starts, so all we drive onto the CALOUT line is DC.  Then, set
       the module to not drive CALOUT onto the VXI sumbus, so we can't
       accidentally mess up anyone else's use of the VXI sumbus.  Then
       start and stop a measurement.  This gets the correct signal
       path loaded into the source, without glitching the output.
       Then do the auto-zero, then restore the sumbus setting, the
       source output mode, and the arm mode.

       You could do this procedure for all auto-zeros, even when the
       filter frequency does not change.  It takes a little longer,
       though. */

    /* Set manual arm, output grounded, sumbus off */
    CHECK(e1432_set_arm_mode(hw, group, E1432_MANUAL_ARM));
    CHECK(e1432_set_source_output(hw, group, E1432_SOURCE_OUTPUT_GROUNDED));
    CHECK(e1432_set_sumbus(hw, group, E1432_SUMBUS_OFF));
    /* Start and stop measurement, then auto-zero */
    CHECK(e1432_init_measure(hw, group));
    CHECK(e1432_reset_measure(hw, group));
    CHECK(e1432_auto_zero(hw, group));
    /* Restore desired sumbus, output, and arm settings */
    CHECK(e1432_set_sumbus(hw, group, E1432_SUMBUS_OFF));
    CHECK(e1432_set_source_output(hw, group, E1432_SOURCE_OUTPUT_NORMAL));
    CHECK(e1432_set_arm_mode(hw, group, E1432_AUTO_ARM));
    return 0;
}

static int
run(E1432ID hw, int group)
{
    /* This starts the source, so we can examine the output DC level
       and confirm that the auto-zero worked properly. */
    CHECK(e1432_init_measure(hw, group));
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, nmod, nchan, group;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:uV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group) < 0)
	return EXIT_FAILURE;
    if (run(hw, group) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
